package com.beiming.basic.chat.api;


import com.beiming.basic.chat.api.dto.request.*;
import com.beiming.basic.chat.api.dto.response.*;
import com.beiming.framework.domain.DubboResult;
import com.beiming.framework.page.PageInfo;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * 聊天、视频服务
 * 
 * @author lb
 *
 */
@Valid
public interface RoomApi {

  /**
   * 新建对话聊天室
   * 
   * @param dto
   * @return roomId
   */
  DubboResult<String> newChatRoom(@Valid ChatRoomReqDTO dto);

  /**
   * 新建对话聊天室（默认房间状态为进行中）
   * 
   * @param dto
   * @return roomId
   */
  DubboResult<String> newChatRoomRunning(ChatRoomReqDTO roomRequestDTO);
  
  /**
   * 新建视频房间
   * 
   * @param dto
   * @return roomId
   */
  DubboResult<String> newVideoRoom(@Valid VideoRoomReqDTO dto);

  /**
   * 新建视频房间，并限制开始结束时间
   * 
   * @param roomRequestDTO
   * @param startTime 开始时间
   * @param endTime 结束时间
   * @return roomId
   */
  DubboResult<String> newVideoRoom(@Valid VideoRoomReqDTO dto,
      @NotNull(message = ChatApiValidationMessage.ROOM_START_TIME_NOT_NULL) Date startTime,
      @NotNull(message = ChatApiValidationMessage.ROOM_END_TIME_NOT_NULL) Date endTime);

  /**
   * 获取房间信息
   * 
   * @param roomId 房间id
   * @return
   */
  DubboResult<RoomResDTO> getRoom(
      @Valid @NotBlank(message = ChatApiValidationMessage.ROOM_ID_NOT_BLANK) String roomId);


  /**
   * 发送文本消息
   * 
   * @param dto
   */
  DubboResult<MessageResDTO> sendTextMessage(@Valid SendTextMessageReqDTO dto);

  /**
   * 发送系统类消息
   *
   * @param dto
   */
  DubboResult sendSystemMessage(@Valid SendSystemMessageReqDTO dto);

  /**
   * 发送文件
   * 
   * @param dto
   */
  DubboResult<MessageResDTO> sendFileMessage(@Valid SendFileMessageReqDTO dto);
  /**
   * 发送诉讼事项
   *
   * @param dto
   */
  DubboResult<MessageResDTO> sendActionMessage(@Valid SendActionMessageReqDTO dto);

  /**
   * 发送诉讼事项
   *
   * @param dto
   */
  DubboResult<MessageResDTO> sendVoiceMessage(@Valid SendVoiceMessageReqDTO dto);
  /**
   * 分页获取聊天消息
   * 
   * @param roomId 房间id
   * @param memberId 成员id
   * @return
   */
  DubboResult<PageInfo<MessageResDTO>> listMessage(@Valid ListMessageReqDTO dto);

  /**
   * 流式分页获取聊天消息
   *
   * @param roomId 房间id
   * @param memberId 成员id
   * @return
   */
  DubboResult<PageInfo<MessageResDTO>> streamListMessage(@Valid StreamListMessageReqDTO dto);


  /**
   * 获取单个房间内，单个成员的聊天消息
   * 
   * @param roomId 房间id
   * @param memberId 成员id
   * @return
   */
  DubboResult<ArrayList<MessageResDTO>> listAllMessage(@Valid ListAllMessageReqDTO dto);

  /**
   * 获取单条消息
   *
   * @param dto
   * @return
   */
  DubboResult<MessageResDTO> loadMessage(@Valid LoadMessageReqDTO dto);
  /**
   * 追加成员
   * 
   * @param dto
   * @return
   */
  DubboResult<RoomResDTO> appendMember(@Valid AppendMemberDTO dto);

  /**
   * 进入房间
   * 
   * @param dto
   * @return
   */
  DubboResult<RoomResDTO> joinRoom(@Valid JoinRoomReqDTO dto);

  /**
   * 退出房间
   * 
   * @param dto
   * @return
   */
  DubboResult<RoomResDTO> quitRoom(@Valid RoomMemberReqDTO dto);

  /**
   * 结束房间
   * 
   * @param dto
   * @return
   */
  DubboResult<Boolean> closeRoom(@Valid RoomMemberReqDTO dto);

  /**
   * 保存混流视频信息
   * 
   * @return
   */
  DubboResult<Boolean> saveRoomMixVideoInfo(@Valid SaveRoomVideoInfoReqDTO dto);

  /**
   * 保存混流视频信息(新版本获取信息是根据腾讯的roomid查询房间信息，区别于老版本直接根据room_info表中的主键查询)
   *
   * @return
   */
  @RequestMapping(value = "saveNewRoomMixVideoInfo", method = RequestMethod.POST)
  DubboResult<Boolean> saveNewRoomMixVideoInfo(@RequestBody @Valid SaveRoomVideoInfoReqDTO dto);


  /**
   * 保存单个成员视频信息
   * 
   * @return
   */
  DubboResult<Boolean> saveRoomMemberVideoInfo(@Valid SaveMemberVideoInfoReqDTO dto);

  /**
   * 删除视频人员信息
   * 
   * @return
   */
  DubboResult<Boolean> deleteRoomMember(@Valid DeleteMemberReqDTO dto);

  /**
   * 发送视频流信息
   * 
   * @param streamInfo
   * @return
   */
  DubboResult<Boolean> sendStreamInfo(@Valid StreamInfo streamInfo);
  
  /**
   * 根据成员iD查询所属的50个房间(未提交0122)
   * @param memberId
   * @author ljf
   * @return
   */
  DubboResult<ArrayList<String>> listRoomIdsByMemberId(String memberId);

  /**
   * 根据subject_id查出房间id和用户
   * @param subjectId
   * @author cjt
   * @return
   */
  DubboResult<ArrayList<MemberRoomsResDTO>> queryAllRoomInfo(String subjectId);
  /**
   * 根据subject_id查出房间id和用户(包含房间主键ID)
   * @param subjectId
   * @author ljf
   * @return
   */
  DubboResult<ArrayList<MemberRoomsNewResDTO>> queryAllNewRoomInfo(String subjectId);
  
  /**
   * 更新成员状态
   * 
   * @return
   */
  DubboResult<Boolean> updateMemberStatus(@Valid UpdateMemberStatusReqDTO dto);

  /*
  查询2个人员是否在同个私聊室
   */
  DubboResult<Boolean> isExistRoom(@Valid QueryExistRoomReqDTO reqDTO);
  /**
   * 根据subject_id和成员ID查出此成员所在的房间id和用户(包含房间主键ID)
   * @param subjectId
   * @param memberId
   * @author ljf
   * @return
   */
  DubboResult<ArrayList<MemberRoomsNewResDTO>> queryAllNewRoomInfo(String subjectId, String memberId);

  
  /**
   * 根据业务类型业务id查询房间
   * 
   * @param subjectType
   * @param subjectId
   * @return
   */
  DubboResult<RoomInfoResDTO> getRoomBySuject(String subjectType, Long subjectId);
  /**
   * 证据归纳查询接口
   * @param dto
   * @return
   */
  DubboResult<PageInfo<MessageResDTO>> inductionEvidence(InductionEvidenceReqDTO dto);

  List<String> checkMemberDuplicate(List<MemberReqDTO> members);
 
  DubboResult<RoomResDTO> getRoomByRealRoomId(String realRoomId);

  /**
   * 撤回消息
   *
   * @param dto dto
   * @return {@link DubboResult}<{@link Integer}>
   */
  DubboResult<Integer> revokeMsg(RevokeMsgDto dto);

  /**
   * chat_info_more表中添加已读用户id
   *
   * @param id      id
   * @param userIds 用户id
   */
  void addUserId(Long id, String userIds);


  DubboResult<ArrayList<MessageMoreResDTO>> getMessageBySubjectId(Long roomId);

  DubboResult<ArrayList<MaterialsMessageResDTO>> getEvidenceMessageBySubjectId(ArrayList<Long> ids);

  DubboResult<ArrayList<RoomInfoResDTO>> getChatRoomBySubject(Long subjectId);

  /**
   * 查询聊天消息中文件的数据权限
   * @param chatInfoId
   * @return
   */
  DubboResult<EvidencePermissionResDTO> getChatInfoPermissionByChatId(Long chatInfoId);

  /**
   * 编辑证据材料可见权限
   * @param editEvidencePermissionDTO
   * @return
   */
  DubboResult<Boolean> editEvidencePermission(@Valid EditEvidencePermissionDTO editEvidencePermissionDTO);
}
