package com.beiming.odr.document.interceptor;

import com.beiming.framework.domain.DubboResult;
import com.beiming.framework.domain.DubboResultBuilder;
import com.beiming.framework.enums.DubboResultCodeEnums;
import com.beiming.framework.exception.DubboBusinessException;
import com.beiming.framework.util.AssertUtils;
import com.beiming.framework.util.AssertUtils.AssertionException;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.springframework.core.Ordered;
import org.springframework.stereotype.Component;

@Aspect
@Component
@Slf4j
public class DubboInterceptor implements Ordered {

  /**
   * 拦截Referee自身
   * 
   * @param point
   * @return
   * @throws Throwable
   */
  @Around("execution(com.beiming.framework.domain.DubboResult com.beiming.odr.document.api..*(..))")
  public Object aroundSelf(ProceedingJoinPoint point) throws Throwable {
    try {
      return point.proceed();
    } catch (Exception e) {
      int code = DubboResultCodeEnums.INTERNAL_ERROR.value();
      String message = e.getMessage();
      if (e instanceof AssertionException) {
        AssertionException ae = (AssertionException) e;
        code = ae.getResultCode().value();
      } else if (e instanceof DubboBusinessException) {
        DubboBusinessException ae = (DubboBusinessException) e;
        code = ae.getCode();
        message = ae.getBusinessMessage();
      }
      log.error("dubbo error : ", e);
      return DubboResultBuilder.error(code, message);
    }
  }

  /**
   * 拦截其他调用
   * 
   * @param point
   * @return
   * @throws Throwable
   */
  @Around("execution(com.beiming.framework.domain.DubboResult *.*(..)) && !execution(com.beiming.framework.domain.DubboResult com.beiming.odr.document.api..*(..))")
  public Object aroundOther(ProceedingJoinPoint point) throws Throwable {
    Object res = point.proceed();
    AssertUtils.assertNotNull(res, DubboResultCodeEnums.INTERNAL_ERROR, "内部系统异常");
    if (res instanceof DubboResult) {
      DubboResult<?> dubboRes = (DubboResult<?>) res;
      if (dubboRes.isSuccess()) {
        return res;
      }
      if (dubboRes.getCode() == DubboResultCodeEnums.PARAM_ERROR.value()) {
        log.info("document服务拦截器报错code{}, message{}", DubboResultCodeEnums.PARAM_ERROR.value(), dubboRes.getMessage());
        throw new DubboBusinessException(DubboResultCodeEnums.PARAM_ERROR, dubboRes.getMessage());
      } else {
        throw new DubboBusinessException(DubboResultCodeEnums.INTERNAL_ERROR,
            dubboRes.getMessage());
      }
    }
    return res;
  }

  @Override
  public int getOrder() {
    return -1;
  }

}
