package com.beiming.odr.referee.enums;

import com.google.common.collect.Lists;
import java.util.List;

/**
 * @Description: 文书类型枚举
 * @Author: caiwei
 * @Date: Created 2018-06-21
 */
public enum DocumentTypeEnum {
  MEDIATION_BOOK("调解协议书"),
  COMMITMENT_BOOK("承诺书"),
  NO_DISSENT_MEDIATION_SCHEME("无异议调解方案"),
  UNDISPUTED_FACT("无争议事实确认协议"),
  MEDIATION_RECORD("调解笔录"),
  INQUIRE_RECORD("调查笔录"),
  DISSENT_RECORD("异议书"),
  LAW_TERMINATED_PROGRESS("正常终结案件调解进度"),
  JUDICIAL_CONFIRM_APPLY_BOOK("司法确认申请书"),
  FINAL_MEDIATION_REPORT("终结调解报告"),
  MEDIATION_APPLY_BOOK("请求法院出具调解书"),
  RECONCILIATION_AGREEMENT("调解书"),
  WITHDRAW_CASE("撤回立案"),
  JUDICIAL_CONFIRM_BOOK("司法确认书"),
  CLOSE_CASE_BOOK("诉中结案文书"),
  COMPRM_SETTLEMENT("和解通知书"),
  REMINDER_LETTER("催告通知书"),
  ADDRESS_CONFIRMATION("地址确认书"),
  APPLICATION_MEDIATION("调解申请书"),
  MEDIATION_ACCEPT_FORM("调解受理登记表"),
  CUSTOM_INSTRUMENTS("自定义文书"),

  ;

  private String name;

  DocumentTypeEnum(String name) {
    this.name = name;
  }

  public static DocumentTypeEnum getDocumentTypeByName(String value) {
    for (DocumentTypeEnum constant : DocumentTypeEnum.values()) {
      if (constant.name().equals(value)) {
        return constant;
      }
    }
    return null;
  }

  /**
   * 判断是否是司法确认申请书
   * @param documentType
   * @return
   */
  public static Boolean isJudicial(String documentType) {
    return DocumentTypeEnum.JUDICIAL_CONFIRM_APPLY_BOOK.name().equals(documentType);
  }

  /**
   * 判断是否为笔录类型文书
   */
  public static Boolean isRecord(String documentType) {
    return DocumentTypeEnum.MEDIATION_RECORD.name().equalsIgnoreCase(documentType)
        || DocumentTypeEnum.INQUIRE_RECORD.name().equalsIgnoreCase(documentType);
  }
  /**
   * 判断是否为当事人优先签的文书
   */
  public static Boolean isPartyFistSign(String documentType) {
    return DocumentTypeEnum.ADDRESS_CONFIRMATION.name().equalsIgnoreCase(documentType)
        || DocumentTypeEnum.APPLICATION_MEDIATION.name().equalsIgnoreCase(documentType);
  }

  public static Boolean needMediatorInfo(DocumentTypeEnum documentType) { //文书需要调解员信息的
    return !DocumentTypeEnum.MEDIATION_RECORD.equals(documentType)
        && !DocumentTypeEnum.INQUIRE_RECORD.equals(documentType)
        && !DocumentTypeEnum.DISSENT_RECORD.equals(documentType)
        && !DocumentTypeEnum.APPLICATION_MEDIATION.equals(documentType);
  }

  public static boolean noNeedSign(DocumentTypeEnum documentType) { //不需要签名的文书
    return DocumentTypeEnum.FINAL_MEDIATION_REPORT.equals(documentType);
  }

  public static boolean onlyApplicantSign(DocumentTypeEnum documentType) { //只有申请人方签名的文书
    return DocumentTypeEnum.APPLICATION_MEDIATION.equals(documentType)
        || DocumentTypeEnum.MEDIATION_ACCEPT_FORM.equals(documentType);
  }

  public static boolean onlyMediatorSign(DocumentTypeEnum documentType) { //只有调解员方签名的文书
    return DocumentTypeEnum.MEDIATION_APPLY_BOOK.equals(documentType);
    // DocumentTypeEnum.JUDICIAL_CONFIRM_APPLY_BOOK.equals(documentType) ||
  }

  public static boolean isSaveAndSend(DocumentTypeEnum documentType) { //保存即发送
    return DocumentTypeEnum.APPLICATION_MEDIATION.equals(documentType);
  }

  public String getName() {
    return name;
  }

  public static List<DocumentTypeEnum> getCaseEndDoc() {
    return Lists.newArrayList(DocumentTypeEnum.JUDICIAL_CONFIRM_APPLY_BOOK,
        DocumentTypeEnum.MEDIATION_APPLY_BOOK);
  }


  /**
   * 判断是否为富文本新文书
   */
  public static Boolean isHtmlBook(DocumentTypeEnum documentType) {
    return DocumentTypeEnum.CUSTOM_INSTRUMENTS.equals(documentType);
  }

}
