package com.beiming.odr.referee.api;

import com.beiming.framework.domain.DubboResult;
import com.beiming.framework.page.PageInfo;
import com.beiming.odr.referee.dto.requestdto.*;
import com.beiming.odr.referee.dto.requestdto.document.InvalidBookReqDTO;
import com.beiming.odr.referee.dto.requestdto.haoda.GetTokenReqDTO;
import com.beiming.odr.referee.dto.responsedto.*;
import com.beiming.odr.referee.enums.DocumentTypeEnum;

import javax.validation.Valid;
import javax.validation.constraints.NotNull;
import java.io.IOException;
import java.util.ArrayList;

/**
 * @Description: 文书api接口
 * @Author: caiwei
 * @Date: Created 2018-06-20
 */
@Valid
public interface DocumentApi {

  /**
   * 判断文书是否存在
   */
  DubboResult<Boolean> checkDocumentExist(Long id);

  /**
   * 获取协议书
   */
  DubboResult<ProtocolBookResDTO> getProtocolBook(ProtocolBookReqDTO reqDTO);

  /**
   * 获取备份协议书
   */
  DubboResult<ProtocolBookResDTO> getTempProtocolBook(ProtocolBookReqDTO reqDTO);

  /**
   * 保存协议书
   */
  DubboResult<ProtocolBookResDTO> saveProtocolBook(@Valid SaveProtocolBookReqDTO dto);

  /**
   * 保存调解笔录
   */
  DubboResult<ProtocolBookResDTO> saveRecordBook(@Valid SaveProtocolBookReqDTO dto);

  /**
   * 保存异议书
   */
  DubboResult<ProtocolBookResDTO> saveDissentBook(@Valid SaveProtocolBookReqDTO dto);

  /**
   * 发送协议书
   */
  DubboResult sendProtocolBook(Long documentId);

  /**
   * 获取调解方案
   */
  DubboResult getMediationScheme(Long caseId);

  /**
   * 保存承诺书
   */
  DubboResult savePromiseBook(SavePromiseBookReqDTO dto);

  /**
   * 拒绝文书
   */
  DubboResult<Long> refuseDocument(
      @Valid @NotNull(message = "{document.id.cannot.be.empty}") Long documentId,
      @Valid @NotNull(message = "{user.id.cannot.be.empty}") Long confirmUserId,
      String userName);

  /**
   * 确认文书<br> 当前用户为该用户类型的最后一人时，进行业务状态处理
   */
  DubboResult<?> confirmDocument(Long documentId, Long userId, String userName, String userType);

  /**
   * 获取案件调解员id
   */
  DubboResult<Long> getMediationIdByCaseId(Long caseId);

  /**
   * 获取司法确认申请书
   */
  DubboResult<JudicialConfirmBookResDTO> getJudicialConfirmBook(Long caseId, DocumentTypeEnum documentTypeEnum, Integer isNew);

  /**
   * 保存司法确认申请书
   */
  DubboResult<Long> saveJudicialConfirmBook(SaveJudicialConfirmBookReqDTO dto);

  /**
   * 发送司法确认申请书
   */
  DubboResult<Long> sendJudicialConfirmBook(SaveJudicialConfirmBookReqDTO dto);

  /**
   * 根据文书id返回案件id
   */
  DubboResult<Long> getCaseIdByDocId(Long id);

  DubboResult sendFinalMediationReport(SendFinalMediationReportReqDTO reqDTO);

  /**
   * 获取调解申请书
   */
  DubboResult<MediationApplyBookResDTO> getMediationApplyBook(Long caseId,
      DocumentTypeEnum documentTypeEnum, Integer isNew);

  /**
   * 保存调解申请书
   */
  DubboResult<Long> saveMediationApplyBook(SaveMediationApplyBookReqDTO dto);

  /**
   * 发送调解申请书
   */
  DubboResult<Long> sendMediationApplyBook(SaveMediationApplyBookReqDTO dto);

  /**
   * 根据案件id查询文书信息
   *
   * @param caseId
   * @param documentType
   * @return
   */
  DubboResult<LawDocumentResDTO> findDocumentByCaseId(Long caseId, String documentType);

  /**
   * 保存网上立案扩展数据信息
   * @param reqDTO
   * @return
   */
  DubboResult<Boolean> saveOnlineFilingExtend(GetTokenReqDTO reqDTO);

  /**
   *  新增文书模板
   */
  DubboResult insertCustomDocMld(CustomDocMldReqDTO docMldReqDTO) throws IOException;

  /**
   *  新增文书模板
   */
  DubboResult<Long> addCustomDocMldById(CommonIdReqDTO reqDTO, Long userId, String userName);

  DubboResult<Long> addPublicDocMldById(CommonIdReqDTO reqDTO, Long userId, String userName);

  /**
   * 获取自定义文书列表
   * @return
   */
  DubboResult<PageInfo<CustomDocMldResDTO>> getCustomDocMldList(CustomDocMldReqDTO dto);

  DubboResult<PageInfo<CustomDocMldResDTO>> getPublicDocMldList(CustomDocMldReqDTO dto);

  /**
   * 获取文书tab列表
   * @return
   */
  DubboResult<ArrayList<CustomDocMldResDTO>> getCustomDocMldTabList(CustomDocMldReqDTO dto);

  /**
   * word转html
   * @return
   */
  DubboResult wordTohtml(WordFileReqDTO dto) throws IOException;

  /**
   * 获取我的收藏文书列表
   * @return
   */
  DubboResult<PageInfo<CustomDocMldResDTO>> getMyCollectDocList(CustomDocMldReqDTO dto);

  /**
   *  文书收藏
   */
  DubboResult addMyCollectDoc(CustomDocMldReqDTO docMldReqDTO);

  /**
   *  取消文书收藏
   */
  DubboResult delMyCollectDoc(CustomDocMldReqDTO docMldReqDTO);

  /**
   * 获取自定义文书内容
   * @return
   */
  DubboResult<CustomDocMldResDTO> getCustomDocMldById(Long id);

  /**
   *  编辑文书模板
   */
  DubboResult editCustomDocMld(CustomDocMldReqDTO docMldReqDTO) throws IOException;

  /**
   *  删除文书模板
   */
  DubboResult delCustomDocMld(CustomDocMldReqDTO docMldReqDTO);

  /**
   * 根据案件id获取文书材料
   * @param caseId
   * @return
   */
  DubboResult<ArrayList<LawDocumentResDTO>> getDocumentByCaseId(Long caseId);

  /**
   * 获取自定义文书列表
   * @param reqDTO
   * @return
   */
  DubboResult<ArrayList<WorkbenchDocStatusResDTO>> getCustomInstruments(ProtocolBookReqDTO reqDTO);

  DubboResult<ProtocolBookResDTO> invalidBook(InvalidBookReqDTO reqDTO);

  /**
   *  共享文书
   */
  DubboResult publicCustomDocMld(CustomDocMldReqDTO docMldReqDTO);

  /**
   *  获取公开文书列表
   */
  DubboResult<com.github.pagehelper.PageInfo<CustomDocMldResDTO>>  getPublicCustomDocList(CustomDocMldReqDTO dto);
}
