package com.beiming.odr.referee.util;

import com.beiming.odr.referee.constant.LocaleConst;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.springframework.context.MessageSource;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Component;

@Component
public class MsgUtils {

  private static MessageSource messageSource;

  public MsgUtils(MessageSource messageSource) {
    MsgUtils.messageSource = messageSource;
  }

  /**
   * 获取单个国际化翻译值
   */
  public static String get(String msgKey, String... args) {
    try {
      return messageSource.getMessage(msgKey, args, LocaleContextHolder.getLocale());
    } catch (Exception e) {
      return msgKey;
    }
  }

  /**
   * 根据语言环境国际化案件编号
   */
  public static String translateCaseNo(String caseNo) {
    return translateCaseNo(caseNo, LocaleContextHolder.getLocale().toString());
  }

  /**
   * 根据语言环境国际化案件编号
   */
  public static String translateCaseNo(String caseNo, String locale) {
    if (caseNo == null || "".equals(caseNo)) {
      return caseNo;
    }
    String localeCaseNo;
    if (caseNo.contains("多元化解")) {
      localeCaseNo = getLocaleCaseNo(locale);
    } else if (caseNo.contains("诉前民调")) {
      localeCaseNo = getLocaleCaseNoBySQMT(locale);
    } else if (caseNo.contains("民诉前调")) {
      localeCaseNo = getLocaleCaseNoByMSQT(locale);
    } else {
//      localeCaseNo = getLocaleCaseNoMC(locale);
      return caseNo;
    }
    return replaceCaseNo(caseNo, localeCaseNo);
  }

  /**
   * 民诉前调 替换案号
   */
  private static String getLocaleCaseNoByMSQT(String locale) {
    String localeCaseNo = "（#year#）粤#no#民诉前调#num#号";
    if (LocaleConst.EN.equalsIgnoreCase(locale)) {
      localeCaseNo = "(#year#) Yue #no# pre-litigation mediation #num#";
    } else if (LocaleConst.TW.equalsIgnoreCase(locale)) {
      localeCaseNo = "（#year#）粵#no#民訴前調#num#號";
    }
    return localeCaseNo;
  }

  /**
   * 替换案号
   */
  private static String replaceCaseNo(String caseNo, String localeCaseNo) {
    //匹配指定范围内的数字
    String regEx = "[^0-9]";
    //Pattern是一个正则表达式经编译后的表现模式
    Pattern p = Pattern.compile(regEx);
    // 一个Matcher对象是一个状态机器，它依据Pattern对象做为匹配模式对字符串展开匹配检查。
    Matcher m = p.matcher(caseNo);
    //将输入的字符串中非数字部分用空格取代并存入一个字符串
    String nums = m.replaceAll(" ").trim();
    nums = nums.replaceAll(" +", " ");
    //以空格为分割符在讲数字存入一个字符串数组中
    String[] numArr = nums.split(" ");
    switch (numArr.length) {
      case 2:
        localeCaseNo = localeCaseNo.replace("#year#", numArr[0]).replace("#num#", numArr[1]);
        break;
      case 3:
        localeCaseNo = localeCaseNo.replace("#year#", numArr[0]).replace("#no#", numArr[1])
            .replace("#num#", numArr[2]);
        break;
      default:
    }

    return localeCaseNo;
  }

  /**
   * 多元化解
   */
  private static String getLocaleCaseNo(String locale) {
    String localeCaseNo = "（#year#）多元化解#num#号";
    if (LocaleConst.EN.equalsIgnoreCase(locale)) {
      localeCaseNo = "(#year#) Diversified Resolution #num#";
    } else if (LocaleConst.TW.equalsIgnoreCase(locale)) {
      localeCaseNo = "（#year#）多元化解#num#號";
    }
    return localeCaseNo;
  }

  /**
   * 诉前民调
   */
  private static String getLocaleCaseNoBySQMT(String locale) {
    String localeCaseNo = "（#year#）粤#no#诉前民调#num#号";
    if (LocaleConst.EN.equalsIgnoreCase(locale)) {
      localeCaseNo = "(#year#) Yue #no# pre-litigation mediation #num#";
    } else if (LocaleConst.TW.equalsIgnoreCase(locale)) {
      localeCaseNo = "（#year#）粵#no#訴前民調#num#號";
    }
    return localeCaseNo;
  }

  /**
   * 民初
   */
  private static String getLocaleCaseNoMC(String locale) {
    String localeCaseNo = "（#year#）粤#no#民初#num#号";
    if (LocaleConst.EN.equalsIgnoreCase(locale)) {
      localeCaseNo = "(#year#) Yue #no# civil first instance #num#";
    } else if (LocaleConst.TW.equalsIgnoreCase(locale)) {
      localeCaseNo = "（#year#）粵#no#民初#num#號";
    }
    return localeCaseNo;
  }
}