package com.beiming.odr.user.api;

import com.beiming.framework.domain.DubboResult;
import com.beiming.framework.page.PageInfo;
import com.beiming.odr.user.api.common.enums.PersonTypeEnum;
import com.beiming.odr.user.api.common.enums.UserLoginTypeEnum;
import com.beiming.odr.user.api.dto.UserInfoDTO;
import com.beiming.odr.user.api.dto.requestdto.*;
import com.beiming.odr.user.api.dto.responsedto.*;

import javax.validation.Valid;
import java.util.ArrayList;
import java.util.HashMap;

@Valid
public interface UserServiceApi {

  /**
   * 新增普通用户
   */
  DubboResult<LoginInfoResDTO> insertCommonUser(
      @Valid CommonUserRegisterReqDTO commonUserRegisterReqDTO);

  /**
   * 用户登录 -可以通过mobile也可以通过loginName
   * 
   * @param mobile
   * @param password
   * @param loginName
   * @param loginType
   */
  DubboResult<LoginInfoResDTO> userLogin(String mobile, String password, String loginName,
      UserLoginTypeEnum loginType);

  /**
   * 更新普通用户
   */
  DubboResult updateCommonUser(CommonUserUpdateReqDTO commonUserUpdateReqDTO);

  /**
   * 重置登录密码
   * 
   * @param commonUserResetPasswordReqDTO
   */
  DubboResult resetUserPassword(@Valid CommonUserResetPasswordReqDTO commonUserResetPasswordReqDTO);

  /**
   * 人脸识别通过-设置
   *
   * @param UserBasicsId 用户ID
   */
  DubboResult setFacialVerify(Long userBasicsId);


  /** 实名认证-设置 */
  DubboResult setRealNameAuthentication(Long userId,
      RealNameAuthenticationReqDTO realNameAuthenticationReqDTO);

  /** 查询普通用户的信息 */
  DubboResult<CommonUserSearchResDTO> searchCommonUser(CommonIdReqDTO commonIdReqDTO);

  /** 查询所有用户的信息-sm4/mac前置查询 */
  DubboResult<ArrayList<CommonUserSearchResDTO>> getAllUserForEncrypt();

  /** 查询所有用户的信息-sm4/mac前置查询 */
  DubboResult<ArrayList<UserRoleRelationResDTO>> getAllUserRoleForEncrypt();

  /**
   * 更新user_basics加密、mac字段
   */
  DubboResult updEncryptForUb(String id, String encUserName, String encIdCard, String encMobile, String iv, String mac);

  /**
   * 更新user_role_relation加密、mac字段
   */
  DubboResult updEncryptForUrr(String id, String encUid, String encRoleCode, String iv, String mac);

  /**
   * 查询服务人员列表分页
   * 
   * @param servicePersonListReqDTO
   * @return
   */
  DubboResult<PageInfo<ServicePersonListResDTO>> getServicePersonListPage(
      ServicePersonListReqDTO servicePersonListReqDTO);

  /**
   * 案件用户注册接口
   */
  DubboResult<CaseUserRegisterResDTO> caseUserregister(
      @Valid CaseUserRegisterReqDTO caseUserRegisterReqDTO);

  /**
   * 修改密码
   * 
   * @param commonUserUpdatePasswordReqDTO
   */
  DubboResult updatePassword(CommonUserUpdatePasswordReqDTO commonUserUpdatePasswordReqDTO);

  /**
   * 修改手机号码
   */
  DubboResult updateMobilePhone(Long userId, String mobilePhone);

  /**
   * 修改邮箱
   */
  DubboResult updateEmail(Long userId, String email);

  /**
   * 修改会议邮箱
   */
  DubboResult updateMeetEmail(Long userId, String email, Long meetingId);
  
  /**
   * 用过手机号码判断用户是否存在
   * 
   * @param mobile 手机号码
   * @param personTypeEnum 人员类型
   * @return true 存在 false 不存在
   */
  DubboResult<Boolean> checkUserByMobile(String mobile, PersonTypeEnum personTypeEnum);

  /**
   * 获取用户角色信息
   * 
   * @param userId 用户id
   * @param orgId 机构id
   */
  DubboResult<UserRoleInfoListResDTO> getRoleInfoByUserId(Long userId);


  /**
   * 获取内部测试用户userId
   * 
   * @return
   */
  DubboResult<String> getInternalTestUserId();

  /**
   * 保存内部测试用户userId,id以逗号隔开
   * 
   * @return
   */
  DubboResult saveInternalTestUserId(String userId);

  /**
   * 获取内部测试机构id
   * 
   * @return
   */
  DubboResult<String> getInternalTestOrgId();

  /**
   * 保存内部测试机构id,id以逗号隔开
   * 
   * @return
   */
  DubboResult saveInternalTestOrgId(String orgId);

  /**
   * 校验姓名和身份证是否匹配
   * 
   * @return
   */
  DubboResult<Boolean> checkRealNameAuthentication(RealNameAuthenticationReqDTO req);

  /**
   * 根据区域代码查询下面的所有调解员 只需要传区域代码
   * 
   * @param arg0
   * @return
   */
  DubboResult<ArrayList<MediatorSingleInfoResDTO>> selMediatorSingleInfoListByAreaCode(
      @Valid SelMediatorSingleInfoReqDTO arg0);

  /**
   * <pre>
   * 根据机构号查询该机构下的所有调解员 只需要传机构代码
   * 
   * @param arg0
   * @return
   */
  DubboResult<ArrayList<MediatorSingleInfoResDTO>> selMediatorSingleInfoListWithOrgManage(
      @Valid SelMediatorSingleInfoReqDTO arg0);

  /***
   * 删除 用户密码是默认8个8，确已经登录的用户（app和小程序）
   */
  DubboResult<ArrayList<Long>> deleteUserLoginToken();


  /**
   * 用户登录 -无需密码
   * 
   * @param UserLoginNotPasswordReqDTO req
   */
  DubboResult<LoginInfoResDTO> userLoginNotPassword(UserLoginNotPasswordReqDTO req);

  /**
   * 微信登录，没有则自动注册账号
   * 
   * @param req
   * @return
   */
  public DubboResult<LoginInfoResDTO> userLoginByWechat(UserLoginByWechatReqDTO req);

  DubboResult setRegisterOrigin(Long userId);

  /**
   * 根据id查询
   * @param id
   * @return
   */
  DubboResult<UserInfoDTO> searchUserInfoByUserId(Long id);
  /**
   * 根据手机号查询
   * 
   * @param mobile
   * @return
   */
  DubboResult<LoginInfoResDTO> userLoginTwo(String mobile);
  
  /**
   * 根据手机号和登录用户类型查询
   * 
   * @param mobile
   * @return
   */
  DubboResult<LoginInfoResDTO> userLoginByMobileAndType(String mobile, UserLoginTypeEnum loginType);
  /**
   * 根据手机号和登录用户类型查询
   *
   * @param mobile
   * @return
   */
  DubboResult<LoginInfoResDTO> userLoginByMobileAndTypes(String mobile, PersonTypeEnum loginType);

  /**
   * 用过手机号码判断用户是否存在
   * (新项目改造,不需要用户类别,是通过角色区分)
   * @param mobile 手机号码
   * @return true 存在 false 不存在
   */
  DubboResult<Boolean> checkUserByMobilePhone(String mobile);

  /**
   * 查询某机构下网格员信息
   * @param orgId 机构id
   * @param userId 过滤人员id
   * @param roleCode 身份code
   * @return
   */
  DubboResult<ArrayList<HashMap<String, Long>>> getGridMemberByOrgId(Long orgId, Long userId, String roleCode);

  /**
   * 查询手机号是否存在
   * @param mobilePhone
   * @return
   */
  DubboResult<Integer> selectMobilePhone(String mobilePhone);

  /**
   * 查询手机号是否存在
   *
   * @param mobilePhone
   * @param personType
   * @return
   */
  DubboResult<Integer> selectMobilePhoneAndPersonType(String mobilePhone, String personType);

  /**
   * 修改手机号、身份证号
   * @param userBasicReqDTO
   */
  void updateByPrimaryKeyIdCard(UserBasicReqDTO userBasicReqDTO);

  /**
   * 搜索区域信息
   *
   * @param areaReqDTO 区要求dto
   * @return {@link DubboResult}<{@link ArrayList}<{@link UserAreasResDTO}>>
   */
  DubboResult<ArrayList<UserAreasResDTO>> searchAreasInfo(UserAreaReqDTO areaReqDTO);

  /**
   * 东湖访客预约系统生成loginInfo
   * @param reqDTO
   * @return
   */
  DubboResult<LoginInfoResDTO> visitorSystemLoginInfo(VisitorSystemUserLoginReqDTO reqDTO);

  /**
   * 东湖访客预约系统 后台法官登录 -可以通过mobile也可以通过loginName
   *临时方案，后面会与数字法院门户对接
   * @param mobile
   * @param password
   * @param loginName
   * @param loginType
   */
  DubboResult<LoginInfoResDTO> userLoginTest(String mobile, String password, String loginName,
      UserLoginTypeEnum loginType);

  /**
   * 东湖访客预约系统 后台法官账号新增
   * @param dto
   * @return
   */
  DubboResult<Boolean> addJudgeUser(AddVisitSystemJudgeReqDTO dto);

  /**
   * 东湖访客预约系统 账号管理列表
   * @param reqDTO
   * @return
   */
  DubboResult<PageInfo<JudgeUserManageResDTO>> judgeUserList(JudgeUserListReqDTO reqDTO);

  /**
   * 东湖访客预约系统 更新法官账号状态
   * @param reqDTO
   * @return
   */
  DubboResult<Boolean> updateJudgeUserStatus(JudgeUserStatusReqDTO reqDTO);

  /**
   * 东湖访客预约系统 查询法官账号详情
   * @param reqDTO
   * @return
   */
  DubboResult<JudgeUserDetailResDTO> getJudgeUserDetails(JudgeUserStatusReqDTO reqDTO);

  /**
   * 根据平台区划代码查询组织机构相关org_id
   * @param areaCode
   * @return
   */
  DubboResult<ArrayList<Long>> searchOrgIdList(String areaCode);


  /**
   * 根据user_id获取用户手机号码
   * @param
   * @return
   */
  DubboResult<UserBasicsResDto> selectById(Long userId);

  /**
   * 根据用户姓名。手机号、人员类型查找工作人员
   * @return
   */
  DubboResult<ArrayList<StaffUserRoleRelationResDTO>> selectUserByNameAndPhone(GetStaffPersonReqDTO reqDTO);

  /**
   * 校验用户的mac值和基本信息加密值
   * @param userId
   * @return
   */
  DubboResult verifyMacAndDecrypt(Long userId, String mobilePhone);

  /**
   * 用户user_basics表 user_name、id_card、mobile_phone字段进行SM4加密 和 mac生成
   * @param userId
   * @param mobilePhone
   * @return
   */
  DubboResult encryptUserBasicAndMac(Long userId, String mobilePhone);

  /**
   * 校验用户角色的mac值和基本信息加密值
   * @return
   */
  DubboResult verifyUserRoleRelationMacAndDecrypt(Long userId, Long userRelationId);

  /**
   * 用户user_role_relation表 user_id、role_code字段进行SM4加密 和 mac生成
   * @param userId
   * @param userRelationId
   * @return
   */
  DubboResult encryptUserRoleRelationAndMac(Long userId, Long userRelationId);
}
