/**
 * 
 */
package com.beiming.odr.user.api.common.utils;

import java.util.regex.Matcher;
import java.util.regex.Pattern;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import com.beiming.framework.context.AppNameContextHolder;
import com.beiming.framework.enums.AppNameEnums;
import com.beiming.framework.enums.DubboResultCodeEnums;
import com.beiming.framework.util.AssertUtils;
import com.beiming.odr.user.api.common.constants.UserConst;
import com.beiming.odr.user.api.common.enums.PasswordFormatTypeEnum;

/**
 * 通用的密码验证类
 * 
 * @author
 *
 */
@Slf4j
public class CommonPasswordValidateUtil {

  /**
   * 核对密码是否正确
   * 
   * @param password 密码
   * @param encodePassWord 加密后的密码
   */
  public static Boolean checkPasswordFlag(String password, String encodePassWord) {
    if (AppNameEnums.YNODR.toString().equalsIgnoreCase(AppNameContextHolder.getAppName())) {
      // 云南的解密
      return Encrypt.matches(password, encodePassWord);
    } else {
      return PassWordUtil.matches(password, encodePassWord);
    }
  }
  
  /**
   * 加密
   * 
   * @param passWord 密码
   * @return
   */
  public static String encodePassWord(String passWord) {
    if (AppNameEnums.YNODR.toString().equalsIgnoreCase(AppNameContextHolder.getAppName())) {
      // 云南的加密
      return Encrypt.md5(passWord);
    } else {
      return PassWordUtil.encodePassWord(passWord);
    }
  }
  
  /**
   * 
   * 核对密码的格式是否正确
   * 
   * @param format  密码规则
   * @param password  密码
   * @param errorMessage  错误内容
   * @return
   */
  public static void checkPasswordFormat(PasswordFormatTypeEnum format, String password, String errorMessage) {
    log.info("密码格式为{}", format.getName());
    AssertUtils.assertTrue(StringUtils.isNotBlank(password), DubboResultCodeEnums.PARAM_ERROR, errorMessage);
    if (AppNameEnums.YNODR.toString().equalsIgnoreCase(AppNameContextHolder.getAppName())) {
      // 云南odr
      checkRegexFormat(password, errorMessage, UserConst.REGEX_PASSWORD_YNODR);
    } else if (AppNameEnums.GUIYANGODR.toString().equalsIgnoreCase(AppNameContextHolder.getAppName())) {
      //贵阳司法局ODR
      checkRegexFormat(password, errorMessage, UserConst.REGEX_PASSWORD_GUIYANGODR);
    } else if (AppNameEnums.HUAIANODR.toString().equalsIgnoreCase(AppNameContextHolder.getAppName())) {
      //淮安ODR
      checkRegexFormat(password, errorMessage, UserConst.REGEX_PASSWORD);
    } else {
      // 不作特别说明：默认石景山
      String reg = "";
      if (PasswordFormatTypeEnum.USER_LOGIN.equals(format)) {
        reg = UserConst.REGEX_PASSWORD;
      } else {
        log.info("重置密码，设置密码格式");
        reg = UserConst.REGEX_PASSWORD_REGISTER;
      }
      checkRegexFormat(password, errorMessage, reg);
    }
  }
  
  /***
   * 检查正则表达式是否匹配
   */
  private static void checkRegexFormat(String password, String message, String regex) {
    Pattern pattern = Pattern.compile(regex);
    Matcher matcher = pattern.matcher(password);
    AssertUtils.assertTrue(matcher.matches(), DubboResultCodeEnums.PARAM_ERROR, message);
  }
  
  /**
   * 设置默认的密码
   * @param  密码
   * @return
   */
  public static String setDefaultPassword() {
    if (AppNameEnums.YNODR.toString().equalsIgnoreCase(AppNameContextHolder.getAppName())) {
      // 云南的加密-云南暂时还没有用，先设置一个
      return UserConst.COMMON_PASSWORD_MD5;
    } else { //如果没有特别说明：都和石景山的密码一样： 默认8个8
      return UserConst.COMMON_PASSWORD_MD5;
    }
  }

}
