package com.beiming.pigeons.api.consumer.hessian;

import static com.beiming.pigeons.api.discover.ServiceDiscoverHelper.BASE_PATH;
import static com.beiming.pigeons.api.discover.ServiceDiscoverHelper.DEFAULT_CONSUMER_NAME;
import static com.beiming.pigeons.api.discover.ServiceDiscoverHelper.NAMESPACE;

import com.alibaba.fastjson.JSONObject;
import com.beiming.framework.domain.DubboResult;
import com.beiming.framework.domain.DubboResultBuilder;
import com.beiming.framework.enums.DubboResultCodeEnums;
import com.beiming.framework.exception.DubboBusinessException;
import com.beiming.framework.log.ActionLog;
import com.beiming.framework.log.ActionLoggerImpl;
import com.beiming.framework.util.RequestIdUtils;
import com.beiming.pigeons.api.consumer.ConsumerAdapter;
import com.beiming.pigeons.api.consumer.ReceiverMessageDto;
import com.beiming.pigeons.api.utils.HessianSerializeUtils;
import com.beiming.pigeons.api.utils.InetUtils;
import java.io.IOException;
import java.io.Serializable;
import java.lang.reflect.Method;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.text.MessageFormat;
import org.apache.commons.lang3.StringUtils;
import org.apache.curator.RetryPolicy;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.retry.ExponentialBackoffRetry;
import org.apache.curator.utils.CloseableUtils;
import org.apache.curator.x.discovery.ServiceDiscovery;
import org.apache.curator.x.discovery.ServiceDiscoveryBuilder;
import org.apache.curator.x.discovery.ServiceInstance;
import org.apache.curator.x.discovery.UriSpec;
import org.apache.curator.x.discovery.details.JsonInstanceSerializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.remoting.caucho.HessianServiceExporter;

/**
 */
public class DefaultConsumerAdapter implements ConsumerAdapter, ApplicationContextAware,
    DisposableBean {

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultConsumerAdapter.class);
  private final ActionLoggerImpl actionLogger = ActionLoggerImpl.get();
  private static ApplicationContext applicationContext;

  private String appPort;

  //    127.0.0.1:2181
  private String zookeeperAddress;

  private String appName;

  private String contextPath;

  CuratorFramework client;

  ServiceDiscovery<Void> serviceDiscovery;


  @Override
  public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
    DefaultConsumerAdapter.applicationContext = applicationContext;
    serviceRegister();

  }

  private void serviceRegister() {
    if (StringUtils.isNotEmpty(zookeeperAddress)) {
//            executorService = Executors.newSingleThreadExecutor();
//
//            executorService.execute(new Runnable() {
//                @Override
//                public void run() {
      try {
        if (zookeeperAddress != null && appPort != null && appName != null) {
          RetryPolicy retryPolicy = new ExponentialBackoffRetry(1000, 3);
          client = CuratorFrameworkFactory.builder()
              .connectString(zookeeperAddress)
              .sessionTimeoutMs(5000)
              .connectionTimeoutMs(10000)
              .retryPolicy(retryPolicy)
              .namespace(NAMESPACE).build();
          client.start();
//                            client.blockUntilConnected();

          serviceStart();
          LOGGER.info("服务注册启动完成============");

        }
      } catch (Exception e) {
        LOGGER.error("zookeeper 服务注册 启动失败=================", e);
      }
//                }
//            });
    }

  }

  private void serviceStart() throws Exception {
    String[] beanNames = applicationContext.getBeanNamesForType(HessianServiceExporter.class);
    String beanName = null;
    if (beanNames == null || beanNames.length == 0) {
      LOGGER.error("没有注册消息接收者====================");
    }
    if (beanNames.length == 1) {
      beanName = beanNames[0];
    } else {
      for (String beanNameTemp : beanNames) {
        if (beanNameTemp.equals(DEFAULT_CONSUMER_NAME)) {
          beanName = DEFAULT_CONSUMER_NAME;
          break;
        }
      }
    }
    if (beanName == null) {
      LOGGER.error("没有找到合适的消息接收bean=================");
    } else {
      if (!beanName.startsWith("/")) {
        beanName = "/" + beanName;
      }
      String contextPathTemp = contextPath != null ? contextPath : "";
      if (!contextPathTemp.startsWith("/") && !StringUtils.isEmpty(contextPathTemp)) {
        contextPathTemp = "/" + contextPathTemp;
      }
      String ip = InetUtils.getLocalHostIp();
      String uri = "http://" + ip + ":" + appPort + contextPathTemp + beanName;
      ServiceInstance<Void> instance = ServiceInstance
          .<Void>builder()
          .name(appName)
          .uriSpec(new UriSpec(uri))
          .build();

      JsonInstanceSerializer<Void> serializer = new JsonInstanceSerializer<Void>(
          Void.class);
      serviceDiscovery = ServiceDiscoveryBuilder
          .builder(Void.class)
          .client(client)
          .serializer(serializer)
          .thisInstance(instance)
          .basePath(BASE_PATH)
          .build();
      serviceDiscovery.start();
      LOGGER.info("本服务的消息接收地址为:" + uri);
//                            ServiceProvider serviceProvider = serviceDiscovery
//                                    .serviceProviderBuilder()
//                                    .serviceName(appName)
//                                    .build();
//                            serviceProvider.start();
//                            Collection<ServiceInstance> serviceInstances = serviceProvider.getAllInstances();
//                            System.out.println(serviceInstances);

    }
  }

  public DubboResult<byte[]> receiveMsg(ReceiverMessageDto messageDto) {
    DubboResult<byte[]> result;
    RequestIdUtils.setRequestId(messageDto.getRequestId());
    try {
      Object service = applicationContext.getBean(messageDto.getServiceName());

      if (service != null) {
        result = doInvoke(messageDto, service);
      } else {
        result = DubboResultBuilder.error(DubboResultCodeEnums.SOURCE_NOT_FOUND.value(),
            MessageFormat.format("There is not bean named:{0}", messageDto.getServiceName()));
      }
    } catch (DubboBusinessException e) {
      result = DubboResultBuilder.error(e.getCode(), e.getBusinessMessage());
      LOGGER.info("Invoke fail result:{}", JSONObject.toJSONString(result));
    } finally {
      RequestIdUtils.removeRequestId();
      actionLogger.save();
    }
    return result;
  }

  private DubboResult<byte[]> doInvoke(ReceiverMessageDto messageDto, Object service) {
    Object methodParam = null;
    try {
      Class clazz = service.getClass();
      methodParam = HessianSerializeUtils.deserialize(messageDto.getParam());
      Method method = clazz.getMethod(messageDto.getMethodName(), methodParam.getClass());
      Object consumerResult = method.invoke(service, methodParam);
      assignActionLogParams(messageDto, consumerResult);
      LOGGER.debug("从消息系统接收消息:调用service:{}, 调用方法:{}, 调用参数：{}, 调用结果:{}", messageDto.getServiceName(),
          messageDto.getMethodName(),
          JSONObject.toJSONString(HessianSerializeUtils.deserialize(messageDto.getParam())),
          JSONObject.toJSONString(consumerResult));
      return buildResult(consumerResult);
    } catch (NoSuchMethodException e) {
      LOGGER.error("method not fond", e);
      throw new DubboBusinessException(DubboResultCodeEnums.SOURCE_NOT_FOUND.value(), MessageFormat
          .format("No method found, method name:{0}, method type:{1}", messageDto.getMethodName(),
              methodParam.getClass()));
    } catch (Throwable t) {
      LOGGER.error("接收消息执行错误, 消息为:" + messageDto, t);
      throw new DubboBusinessException(DubboResultCodeEnums.INTERNAL_ERROR.value(),
          DubboResultCodeEnums.INTERNAL_ERROR.name());
    }
  }

  private void assignActionLogParams(ReceiverMessageDto messageDto, Object consumerResult)
      throws IOException {

    actionLogger.initialize();
    ActionLog actionLog = actionLogger.currentActionLog();
    if (null != actionLog) {
      actionLog.setRequestId(messageDto.getRequestId());
      actionLog.setAction(
          String.format("%s-%s", messageDto.getServiceName(), messageDto.getMethodName()));
      actionLog.setRequestContent(
          JSONObject.toJSONString(HessianSerializeUtils.deserialize(messageDto.getParam())));
      actionLog.setResponseContent(JSONObject.toJSONString(consumerResult));
    } else {
      LOGGER.info("消息系统客户端, actionLog 为空");
    }
  }

  private DubboResult<byte[]> buildResult(Object consumerResult) throws IOException {
    if (consumerResult instanceof Serializable) {
      byte[] resultByteArray = HessianSerializeUtils.serialize(consumerResult);
      return DubboResultBuilder.success(resultByteArray);
    } else {
      return DubboResultBuilder.success(null);
    }
  }

  public void setAppPort(String appPort) {
    this.appPort = appPort;
  }

  public void setZookeeperAddress(String zookeeperAddress) {
    this.zookeeperAddress = zookeeperAddress;
  }

  public void setAppName(String appName) {
    this.appName = appName;
  }

  public void setContextPath(String contextPath) {
    this.contextPath = contextPath;
  }

  public static void main(String[] args) throws UnknownHostException {
    InetAddress addr = InetAddress.getLocalHost();
    String ip = addr.getHostAddress();
    System.out.println(ip);
    System.out.println(InetUtils.getLocalAddress().getHostAddress());
  }


  @Override
  public void destroy() throws Exception {
    if (serviceDiscovery != null) {
      CloseableUtils.closeQuietly(serviceDiscovery);
    }
    if (client != null) {
      CloseableUtils.closeQuietly(client);

    }
  }
}
