package com.beiming.pigeons.api.consumer.rocketmq;

import com.alibaba.fastjson.JSONObject;
import org.apache.rocketmq.client.consumer.DefaultMQPushConsumer;
import org.apache.rocketmq.client.consumer.listener.ConsumeConcurrentlyContext;
import org.apache.rocketmq.client.consumer.listener.ConsumeConcurrentlyStatus;
import org.apache.rocketmq.client.consumer.listener.MessageListenerConcurrently;
import org.apache.rocketmq.client.exception.MQClientException;
import org.apache.rocketmq.common.consumer.ConsumeFromWhere;
import org.apache.rocketmq.common.message.MessageExt;
import org.apache.rocketmq.common.protocol.heartbeat.MessageModel;
import com.beiming.framework.util.RemoteAddressUtil;
import com.beiming.pigeons.api.constants.KangarooConstants;
import com.beiming.pigeons.api.consumer.ConsumeErrorDto;
import com.beiming.pigeons.api.consumer.ConsumeErrorService;
import com.beiming.pigeons.api.exception.KangarooConsumerException;
import com.beiming.pigeons.api.rocketmq.RocketMqAddrService;
import com.beiming.pigeons.api.rocketmq.RocketMqClientDto;
import com.beiming.pigeons.api.utils.InetUtils;
import com.beiming.pigeons.api.utils.JsonSerializeUtil;
import com.beiming.pigeons.api.utils.StopWatch;
import com.beiming.framework.domain.DubboResult;
import com.beiming.framework.enums.RequestTypeEnums;
import com.google.common.collect.Maps;
import com.beiming.framework.domain.PlatformConfig;
import com.beiming.framework.log.ActionLoggerImpl;
import com.beiming.framework.constant.LogConstants;
import com.beiming.framework.log.TraceLogger;
import com.beiming.framework.util.RequestIdUtils;
import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Proxy;
import java.lang.reflect.Type;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentMap;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ClassUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import org.springframework.aop.SpringProxy;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.context.ApplicationListener;
import org.springframework.context.event.ContextRefreshedEvent;


/**
 *
 *
 */
public class RocketConsumerClient implements ApplicationListener<ContextRefreshedEvent>,
    DisposableBean {

  private final Logger logger = LoggerFactory.getLogger(RocketConsumerClient.class);

  private DefaultMQPushConsumer defaultMQPushConsumer;

  private String mqConsumerGroup;

  private String topic;

  private String rocketMqName;

  private RocketMqAddrService rocketMqAddrService;

  /**
   * 如果消费错误，通过调用该service上传错误到消息系统
   */
  private ConsumeErrorService consumeErrorService;

  /**
   * 接收消息实际处理类
   */
  private RocketMsgProcessor processor;

  private static ConcurrentMap<String, DefaultMQPushConsumer> consumers = Maps.newConcurrentMap();

  String localIp = InetUtils.getLocalHostIp();

//    private volatile boolean start = false;

//    private ScheduledExecutorService executorService = Executors.newSingleThreadScheduledExecutor();

  /**
   * @throws MQClientException 通过配置文件初始化参数
   */
  private void init(MessageListenerConcurrently msgListener)
      throws InterruptedException, MQClientException {
    RocketMqClientDto clientDto = new RocketMqClientDto();
    clientDto.setClientIp(localIp);
    clientDto.setClientGroup(mqConsumerGroup);
    clientDto.setRocketMqName(rocketMqName);
    clientDto.setTopic(topic);
    clientDto.setClientType(KangarooConstants.CONSUMER_CLIENT);
    clientDto.setQueueNum(queueNum);
    DubboResult<String> namesrvAddrResult = rocketMqAddrService.getRmNamesrvAddr(clientDto);

    if (!namesrvAddrResult.isSuccess()) {
      logger.error("获取 rocketMq nameServer 地址失败，message=" + namesrvAddrResult.getMessage());
      return;
    }
    String namesrvAddr = namesrvAddrResult.getData();
    if (batchMaxSize == null || batchMaxSize < 1) {
      batchMaxSize = 1;
    }

    if (StringUtils.isEmpty(mqConsumerGroup)
        || StringUtils.isEmpty(namesrvAddr)
        || StringUtils.isEmpty(topic)) {
      logger.error("缺少初始化参数不能为空。。。。。。");
      return;
    }
    // 参数信息
    logger.info("DefaultMQPushConsumer " + mqConsumerGroup + " initialize!");
    logger.info("mqConsumerGroup:" + mqConsumerGroup);
    logger.info("namesrvAddr:" + namesrvAddr);
    logger.info("topic:" + topic);
    logger.info("tag:" + tag);
    logger.info("messageModel:" + getMessageModelByCN(messageModel));
    // 一个应用创建一个Consumer，由应用来维护此对象，可以设置为全局对象或者单例<br>
    // 注意：ConsumerGroupName需要由应用来保证唯一
    if (consumers.containsKey(mqConsumerGroup)) {
      logger.info(mqConsumerGroup + "的 consumer 已经创建，直接返回，建议检查配置是否重复");
      defaultMQPushConsumer = consumers.get(mqConsumerGroup);
      return;
    }
    if (processor == null || consumeErrorService == null) {
      logger.error("消息的客户端没有配置processor或者consumeErrorService");
      return;
    }
    defaultMQPushConsumer = new DefaultMQPushConsumer(mqConsumerGroup);
    if (consumers.putIfAbsent(mqConsumerGroup, defaultMQPushConsumer) != null) {
      logger.info(mqConsumerGroup + "的 consumer 已经创建，直接返回，建议检查配置是否重复");
      defaultMQPushConsumer = consumers.get(mqConsumerGroup);
      return;
    }
    defaultMQPushConsumer.setNamesrvAddr(namesrvAddr);

    // 订阅指定MyTopic下tags的消息，如果tag为空，则订阅所有tag
    if (StringUtils.isEmpty(tag)) {
      tag = "*";
    }
    defaultMQPushConsumer.subscribe(topic, tag);
    // 设置Consumer第一次启动是从队列头部开始消费还是队列尾部开始消费<br>
    // 如果非第一次启动，那么按照上次消费的位置继续消费
    defaultMQPushConsumer.setConsumeFromWhere(ConsumeFromWhere.CONSUME_FROM_LAST_OFFSET);
    // 设置为集群消费(区别于广播消费)
    if (StringUtils.isEmpty(messageModel)) {
      defaultMQPushConsumer.setMessageModel(MessageModel.CLUSTERING);
    } else {
      defaultMQPushConsumer.setMessageModel(getMessageModelByCN(messageModel));
    }

    defaultMQPushConsumer.setConsumeMessageBatchMaxSize(batchMaxSize);
    defaultMQPushConsumer.setVipChannelEnabled(false);
    defaultMQPushConsumer.setUnitName(rocketMqName + "_" + mqConsumerGroup);
    defaultMQPushConsumer.registerMessageListener(msgListener);
    if (consumeThreadMax != null) {
      defaultMQPushConsumer.setConsumeThreadMax(consumeThreadMax);
    }
    if (consumeThreadMin != null) {
      defaultMQPushConsumer.setConsumeThreadMin(consumeThreadMin);
    }
    // Consumer对象在使用之前必须要调用start初始化，初始化一次即可
    defaultMQPushConsumer.start();
    logger.info("DefaultMQPushConsumer: " + mqConsumerGroup + " start success!");
  }

  /**
   * @throws MQClientException
   * @throws InterruptedException
   */
  public void start() throws MQClientException, InterruptedException {
    if (processor == null) {
      throw new KangarooConsumerException("没有处理消息的processor");
    }
    this.init(new MessageListenerConcurrently() {
      @Override
      public ConsumeConcurrentlyStatus consumeMessage(List<MessageExt> messageExtList,
          ConsumeConcurrentlyContext consumeConcurrentlyContext) {
        if (CollectionUtils.isEmpty(messageExtList)) {
          return ConsumeConcurrentlyStatus.CONSUME_SUCCESS;
        }
        for (MessageExt messageExt : messageExtList) {
          consumeMessage(messageExt);
        }
        return ConsumeConcurrentlyStatus.CONSUME_SUCCESS;
      }

      private void consumeMessage(MessageExt messageExt) {
        String msgKey = messageExt.getKeys();
        boolean retry = false;
        String errorReason = "";
        TraceLogger.get().initialize();
        ActionLoggerImpl actionLogger = initActionLogger(messageExt, PlatformConfig.getPlatform());
        StopWatch stopWatch = new StopWatch();
        logger.info("======= begin request processing =======");
        try {
          errorReason = processMsg(messageExt, actionLogger);
          logger.info("=======" + errorReason + "=======");
          if (StringUtils.isNotBlank(errorReason)) {
            if (errorReason.contains("成功") || errorReason.contains("success")) {
              logger.info("如果消息处理成功，topic={}, msgKey={}，方法不需要返回处理结果, 处理结果为={}", topic, msgKey,
                  errorReason);
            } else if ("SmsAndPushMsgTopic".equalsIgnoreCase(topic) && errorReason.contains("模板")) {
              logger.info("消息处理失败，topic={}, msgKey={}, errorReason={}", topic, msgKey, errorReason);
            } else {
//              LogUtils.logWarning(errorReason);
              dealConsumeResult(messageExt, errorReason, retry);
            }
          }
        } catch (Exception e) {
          logger.info(
              mqConsumerGroup + "消费者消费消息失败, msgKey=" + msgKey + ", 错误message=" + e.getMessage(), e);
          errorReason = e.getClass().getSimpleName() + ":" + e.getMessage();
//          LogUtils.logException(e);
          retry = true; //抛异常时自动重新消费
          dealConsumeResult(messageExt, errorReason, retry);
        } finally {
          logger.info("消息处理完毕，处理结果={}，耗时={}ms.", errorReason, stopWatch.elapsedTime());
          logger.info("======= finish request processing =======\r\n");
          actionLogger.currentActionLog().setElapsedTime(stopWatch.elapsedTime());
          TraceLogger.get().cleanup(true);
          //删除当前ThreadLocal中的requestId
          RequestIdUtils.removeRequestId();
          actionLogger.save();
        }
      }
    });
  }

  private <M> String processMsg(MessageExt msg, ActionLoggerImpl actionLogger)
      throws ClassNotFoundException {
    Class originClazz = processor.getClass();
    if (processor instanceof SpringProxy) {
      originClazz = processor.getClass().getSuperclass();
      if (originClazz.equals(Proxy.class)) {
        String processorProxyName = processor.toString();
        String originClazzName = processorProxyName.split("@")[0];
        originClazz = Class.forName(originClazzName);
      }
    }
    Type[] types = originClazz.getGenericInterfaces();
    Class<M> clazz = null;
    for (Type type : types) {
      if (type instanceof ParameterizedType) {
        if (((ParameterizedType) type).getRawType().equals(RocketMsgProcessor.class)) {
          clazz = (Class<M>) ((ParameterizedType) type).getActualTypeArguments()[0];
          break;
        }
      }
    }
    if (clazz != null) {
      String action = ClassUtils.getSimpleName(originClazz);
      actionLogger.currentActionLog().setAction(action);
      MDC.put(LogConstants.ACTION, action);
      M messageData = JsonSerializeUtil.deserialize(msg.getBody(), clazz);
      String messageContent = JSONObject.toJSONString(messageData);
      logger.info("处理消息，处理该消息的类为={}，topic={}, msgKey={}, 消息={}", processor.getClass().getName(),
          topic, msg.getKeys(), messageContent);
      actionLogger.currentActionLog().setRequestContent(messageContent);
      String result = processor.process(messageData);
      if (StringUtils.isNotBlank(result)) {
        logger.error("消息消费结果:{}", result);
      }
      return result;
    } else {
      throw new KangarooConsumerException("json要转换的class为空， 父类的泛型types为" + types.toString());
    }
  }

  private void dealConsumeResult(MessageExt messageExt, String errorReason, boolean retry) {
    if (consumeErrorService == null) {
      logger.info("consumeErrorService为空将不能把错误消息发送到消息系统，请在spring配置文件中配置consumerErrorService");
      return;
    }
    try {
      logger.info("将错误的消息写入数据库，是否自动重试={}", retry);
      ConsumeErrorDto consumeErrorDto = new ConsumeErrorDto();
      consumeErrorDto.setClientIp(localIp);
      consumeErrorDto.setErrorReason(errorReason);
      Field msgIdField = MessageExt.class.getDeclaredField("msgId");
      msgIdField.setAccessible(true);
      String msgId = (String) msgIdField.get(messageExt);
      consumeErrorDto.setMqMsgId(msgId);
      consumeErrorDto.setTopic(messageExt.getTopic());
      consumeErrorDto.setRetry(retry);
      consumeErrorDto.setRocketMqName(rocketMqName);
      consumeErrorDto.setGroupName(mqConsumerGroup);
      consumeErrorDto
          .setRequestId(messageExt.getProperty(KangarooConstants.ROCKET_REQUEST_ID_NAME));
      consumeErrorDto.setMsgSourcePlatform(messageExt.getProperty(KangarooConstants.PLATFORM_NAME));
      consumeErrorService.consumeError(consumeErrorDto);
    } catch (NoSuchFieldException e) {
//      LogUtils.logExtendParams("NoSuchFieldException", e.getMessage());
      logger.error("通过dubbo发送异常消息到消息系统失败", e);
    } catch (IllegalAccessException e) {
//      LogUtils.logExtendParams("IllegalAccessException", e.getMessage());
      logger.error("通过dubbo发送异常消息到消息系统失败", e);
    }
  }

  private ActionLoggerImpl initActionLogger(MessageExt msg, String platform) {
    ActionLoggerImpl actionLogger = ActionLoggerImpl.get();
    actionLogger.initialize();
    String sourcePlatform = msg.getProperty(KangarooConstants.PLATFORM_NAME);
    actionLogger.currentActionLog().setSourcePlatform(sourcePlatform);
    actionLogger.currentActionLog().setCurrentPlatform(platform);
    String requestId = msg.getProperty(KangarooConstants.ROCKET_REQUEST_ID_NAME);
    if (StringUtils.isBlank(requestId)) {
      requestId = RequestIdUtils.generateDefaultRequestId();
    }
    MDC.put("REQUEST_ID", requestId);
    this.logger.info("requestId={}", requestId);
    RequestIdUtils.setRequestId(requestId);
    actionLogger.currentActionLog().setRequestId(requestId);
//        String action = String.format("%s-%s", mqConsumerGroup, topic);
//        actionLogger.currentActionLog().setAction(action);
//        MDC.put(LogConstants.ACTION, action);
    actionLogger.currentActionLog().setClientIP(msg.getBornHostString());
    actionLogger.currentActionLog().setServerIP(new RemoteAddressUtil().getServerIP());
    actionLogger.currentActionLog().setRequestType(RequestTypeEnums.MQ);
    return actionLogger;
  }


  private DefaultMQPushConsumer getDefaultMQPushConsumer() {
    return defaultMQPushConsumer;
  }

  @Override
  public void onApplicationEvent(ContextRefreshedEvent contextRefreshedEvent) {
    Timer timer = new Timer();
    timer.schedule(new TimerTask() {
      @Override
      public void run() {
        try {
          start();
        } catch (Throwable e) {
          logger.error(mqConsumerGroup + "消息系统客户端启动异常", e);
          boolean startError = true;
          for (int i = 0; i < 5 && startError; i++) {
            startError = isStartError(startError, i);
          }
          if (startError) {
            logger.error("5次重新启动consumerClient:" + mqConsumerGroup + "仍然失败");
          }
        }
      }

      private boolean isStartError(boolean startError, int i) {
        try {
          Thread.sleep(3000);
          start();
          startError = false;
          logger.info("重新启动consumerClient" + mqConsumerGroup + ", 重新启动成功");
        } catch (Exception e1) {
          logger.error(mqConsumerGroup + " consumerClient 第" + (i + 1) + "次重新启动失败", e1);
        }
        return startError;
      }
    }, 500);

  }

  /**
   * Spring bean destroy-method
   */
  public void destroy() {
    try {
      if (defaultMQPushConsumer != null) {
        defaultMQPushConsumer.shutdown();

//                synchronized (this) {
//                    start = false;
//                    this.notify();
//                }
      }
    } catch (Throwable t) {
//            t.printStackTrace();
      logger.error("RConsumer defaultMQPushConsumer " + mqConsumerGroup + " destroy exception!", t);
    }
    logger.info("RConsumer defaultMQPushConsumer " + mqConsumerGroup + " destroy success");
  }


  public String getTag() {
    return tag;
  }

  public void setMqConsumerGroup(String mqConsumerGroup) {
    this.mqConsumerGroup = mqConsumerGroup;
  }

  public void setTopic(String topic) {
    this.topic = topic;
  }

  public void setTag(String tag) {
    this.tag = tag;
  }

  public void setBatchMaxSize(int batchMaxSize) {
    this.batchMaxSize = batchMaxSize;
  }

  public void setConsumeThreadMax(Integer consumeThreadMax) {
    this.consumeThreadMax = consumeThreadMax;
  }

  public void setConsumeThreadMin(Integer consumeThreadMin) {
    this.consumeThreadMin = consumeThreadMin;
  }

  public void setMessageModel(String messageModel) {
    this.messageModel = messageModel;
  }


  public void setRocketMqName(String rocketMqName) {
    this.rocketMqName = rocketMqName;
  }

  public void setRocketMqAddrService(RocketMqAddrService rocketMqAddrService) {
    this.rocketMqAddrService = rocketMqAddrService;
  }

  public void setConsumeErrorService(ConsumeErrorService consumeErrorService) {
    this.consumeErrorService = consumeErrorService;
  }

  public void setQueueNum(Integer queueNum) {
    this.queueNum = queueNum;
  }

  public void setBatchMaxSize(Integer batchMaxSize) {
    this.batchMaxSize = batchMaxSize;
  }

  public void setProcessor(RocketMsgProcessor processor) {
    this.processor = processor;
  }

  private MessageModel getMessageModelByCN(String modeCN) {
    MessageModel[] messageModels = MessageModel.values();
    for (MessageModel messageModel : messageModels) {
      if (messageModel.getModeCN().equals(modeCN)) {
        return messageModel;
      }
    }
    return MessageModel.CLUSTERING;
  }

  private String messageModel;


  private String tag;


  private Integer batchMaxSize;


  private Integer consumeThreadMax;

  private Integer consumeThreadMin;

  /**
   * 消费客户端可以自定义队列大小
   */
  private Integer queueNum;
}


