package com.beiming.pigeons.api.discover;

import static com.beiming.pigeons.api.discover.ServiceDiscoverHelper.BASE_PATH;
import static com.beiming.pigeons.api.discover.ServiceDiscoverHelper.KANGAROO_APP_NAME;
import static com.beiming.pigeons.api.discover.ServiceDiscoverHelper.NAMESPACE;

import com.beiming.pigeons.api.exception.KangarooException;
import com.beiming.pigeons.api.producer.ProducerService;
import com.caucho.hessian.client.HessianProxyFactory;
import com.google.common.collect.Maps;
import java.util.Map;
import org.apache.curator.RetryPolicy;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.retry.ExponentialBackoffRetry;
import org.apache.curator.x.discovery.ServiceDiscovery;
import org.apache.curator.x.discovery.ServiceDiscoveryBuilder;
import org.apache.curator.x.discovery.ServiceInstance;
import org.apache.curator.x.discovery.ServiceProvider;
import org.apache.curator.x.discovery.details.JsonInstanceSerializer;
import org.apache.curator.x.discovery.strategies.RandomStrategy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 消息系统发现者，用于客户端发现消息系统
 */
public class KangarooDiscovery {

  private static final Logger LOGGER = LoggerFactory.getLogger(KangarooDiscovery.class);

  //zookeeperClient
  private static CuratorFramework client;

  private static ServiceDiscovery<Void> serviceDiscovery;

  private static ServiceProvider<Void> provider;

  private static Map<String, ProducerService> producerMap = Maps.newHashMap();

  private static HessianProxyFactory proxyFactory = new HessianProxyFactory();

  public static void start(String zookeeperAddress) {
    try {
      RetryPolicy retryPolicy = new ExponentialBackoffRetry(1000, 3);
      client = CuratorFrameworkFactory.builder()
          .connectString(zookeeperAddress)
          .sessionTimeoutMs(10000)
          .connectionTimeoutMs(10000)
          .retryPolicy(retryPolicy)
          .namespace(NAMESPACE).build();
      client.start();
//            client.blockUntilConnected();

      JsonInstanceSerializer<Void> serializer = new JsonInstanceSerializer<Void>(
          Void.class);
      serviceDiscovery = ServiceDiscoveryBuilder
          .builder(Void.class)
          .client(client)
          .serializer(serializer)
          .basePath(BASE_PATH)
          .build();
      serviceDiscovery.start();

      provider = serviceDiscovery
          .serviceProviderBuilder()
          .serviceName(KANGAROO_APP_NAME)
          .providerStrategy(new RandomStrategy<Void>())
          .build();
      provider.start();
    } catch (Throwable t) {
      throw new KangarooException("发现者的类启动失败", t);
    }
  }

  public static ProducerService getProducer() {
    try {
      ServiceInstance<Void> serviceInstance = provider.getInstance();
      if (serviceInstance == null) {
        LOGGER.error("没有找到消息系统的服务注册========");
      }
      String serviceUrl = serviceInstance.getUriSpec().build();
      LOGGER.info("发现消息系统的服务地址为:" + serviceUrl);
      ProducerService producerService = producerMap.get(serviceUrl);
      if (producerService == null) {
        synchronized (KangarooDiscovery.class) {
          producerService = producerMap.get(serviceUrl);
          if (producerService == null) {
            producerService = (ProducerService) proxyFactory
                .create(ProducerService.class, serviceUrl);
            producerMap.put(serviceUrl, producerService);
          }
        }
      }
      return producerService;
    } catch (Throwable t) {
      throw new KangarooException("消息发送者客户端发送异常", t);
    }

  }
}
