package com.beiming.pigeons.api.producer.hessian;

import com.beiming.framework.domain.DubboResult;
import com.beiming.framework.util.RequestIdUtils;
import com.beiming.pigeons.api.discover.KangarooDiscovery;
import com.beiming.pigeons.api.exception.KangarooException;
import com.beiming.pigeons.api.producer.MessageDto;
import com.beiming.pigeons.api.producer.ProducerService;
import com.beiming.pigeons.api.utils.HessianSerializeUtils;
import com.caucho.hessian.client.HessianProxyFactory;
import com.google.common.collect.Maps;
import java.util.Map;
import org.apache.commons.lang3.StringUtils;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.utils.CloseableUtils;
import org.apache.curator.x.discovery.ServiceDiscovery;
import org.apache.curator.x.discovery.ServiceProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;


/**
 */
public class HessianProducerClient implements InitializingBean, DisposableBean {

  private static Logger logger = LoggerFactory.getLogger(HessianProducerClient.class);

  private String serviceUrl;
  private int readTimeout = 5000;

  private String zookeeperAddress;

  CuratorFramework client;

  ServiceDiscovery<Void> serviceDiscovery;

  ServiceProvider<Void> provider;

  private Boolean overloadEnabled = Boolean.valueOf(true);

  private Map<String, ProducerService> producerMap = Maps.newHashMap();

  private HessianProxyFactory proxyFactory = new HessianProxyFactory();
  private Object hessianObject;

  public DubboResult<String> sendMessage(SenderMessageDto senderMessageDto) {

    try {
      ProducerService producerService;
      if (this.hessianObject != null) {
        producerService = (ProducerService) this.hessianObject;
      } else {
        producerService = KangarooDiscovery.getProducer();
      }
      MessageDto messageDto = new MessageDto();
      String requestId = RequestIdUtils.generateNextRequestId();
      messageDto
          .setReceiverParam(HessianSerializeUtils.serialize(senderMessageDto.getReceiverParam()));
      messageDto.setReceiverAddress(senderMessageDto.getReceiverAddress());
      messageDto.setCallbackAddress(senderMessageDto.getCallbackAddress());
      messageDto.setDeliverType(senderMessageDto.getDeliverType());
      messageDto.setKeyword(senderMessageDto.getKeyword());
      messageDto.setTopic(senderMessageDto.getTopic());
      messageDto.setRequestId(requestId);

      return producerService.sendMessage(messageDto);
    } catch (Throwable e) {
      logger.error("发送消息失败", e);
      throw new KangarooException("发送消息失败", e);
    }

  }


  public void afterPropertiesSet()
      throws Exception {
    this.proxyFactory.setChunkedPost(false);
    this.proxyFactory.setOverloadEnabled(this.overloadEnabled.booleanValue());
    this.proxyFactory.setReadTimeout(getReadTimeout());
    if (StringUtils.isEmpty(serviceUrl) && StringUtils.isNotEmpty(zookeeperAddress)) {
      KangarooDiscovery.start(zookeeperAddress);

    } else {
      this.hessianObject = this.proxyFactory.create(ProducerService.class, this.serviceUrl);
    }


  }


  public String getServiceUrl() {
    return this.serviceUrl;
  }

  public void setServiceUrl(String serviceUrl) {
    this.serviceUrl = serviceUrl;
  }

  public void setZookeeperAddress(String zookeeperAddress) {
    this.zookeeperAddress = zookeeperAddress;
  }

  public int getReadTimeout() {
    return this.readTimeout;
  }

  public void setReadTimeout(int readTimeout) {
    this.readTimeout = readTimeout;
  }

  public Boolean getOverloadEnabled() {
    return this.overloadEnabled;
  }

  public void setOverloadEnabled(Boolean overloadEnabled) {
    this.overloadEnabled = overloadEnabled;
  }

  @Override
  public void destroy() throws Exception {
    if (provider != null) {
      CloseableUtils.closeQuietly(provider);
    }
    if (serviceDiscovery != null) {
      CloseableUtils.closeQuietly(serviceDiscovery);
    }
    if (client != null) {
      CloseableUtils.closeQuietly(client);
    }
  }
}
