package com.beiming.pigeons.api.producer.rocketmq;

import com.alibaba.fastjson.JSONObject;
import org.apache.rocketmq.client.exception.MQClientException;
import org.apache.rocketmq.client.producer.DefaultMQProducer;
import org.apache.rocketmq.client.producer.SendResult;
import org.apache.rocketmq.client.producer.SendStatus;
import com.beiming.pigeons.api.constants.DeliverTypeConstants;
import com.beiming.pigeons.api.constants.KangarooConstants;
import com.beiming.pigeons.api.producer.MessageDto;
import com.beiming.pigeons.api.producer.ProducerService;
import com.beiming.pigeons.api.rocketmq.RocketMqAddrService;
import com.beiming.pigeons.api.rocketmq.RocketMqClientDto;
import com.beiming.pigeons.api.utils.InetUtils;
import com.beiming.pigeons.api.utils.JsonSerializeUtil;
import com.beiming.pigeons.api.utils.StopWatch;
import com.beiming.framework.domain.DubboResult;
import com.beiming.framework.domain.DubboResultBuilder;
import com.beiming.framework.enums.DubboResultCodeEnums;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.beiming.framework.domain.PlatformConfig;
import com.beiming.framework.util.RequestIdUtils;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.context.ApplicationListener;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.util.StringUtils;

/**
 *
 */
public class RocketProducerClient implements ApplicationListener<ContextRefreshedEvent>,
    DisposableBean {

  private final Logger logger = LoggerFactory.getLogger(RocketProducerClient.class);

  /**
   * rocketMq集群名
   */
  private String rocketMqName;

  /**
   * rocketMq 消息生产者group名
   */
  private String mqProducerGroup;

  private ProducerService producerService;

  private RocketMqAddrService rocketMqAddrService;

  private DefaultMQProducer defaultMQProducer;

  private volatile boolean isStarted = false;

  private volatile boolean startError = false;

  private static final Set<String> TOPICS = Sets.newConcurrentHashSet();

  /**
   * 防止重复创建producerGroup
   */
  private static final ConcurrentMap<String, DefaultMQProducer> PRODUCERS = Maps.newConcurrentMap();

  String localIp = InetUtils.getLocalHostIp();
//    private static final ScheduledExecutorService executorService = Executors.newSingleThreadScheduledExecutor();

  /**
   * 进行初始化
   */
  private void init() throws MQClientException, InterruptedException {
    RocketMqClientDto clientDto = new RocketMqClientDto();
    clientDto.setClientIp(localIp);
    clientDto.setClientGroup(mqProducerGroup);
    clientDto.setRocketMqName(rocketMqName);
    clientDto.setClientType(KangarooConstants.PRODUCER_CLIENT);

    DubboResult<String> addrResult = rocketMqAddrService.getRmNamesrvAddr(clientDto);
    if (!addrResult.isSuccess()) {
      logger.error(
          "获取 rocketMq：" + rocketMqName + " nameServer 地址失败，message=" + addrResult.getMessage());
      return;
    }
    String namesrvAddr = addrResult.getData();
    if (StringUtils.isEmpty(mqProducerGroup) || StringUtils.isEmpty(namesrvAddr)) {
      logger.error("mqProducerGroup or namesrvAddr parameters is null");
      return;
    }
    // 参数信息
    logger.info("DefaultMQProducer " + mqProducerGroup + " initialize!");
    logger.info("mqProducerGroup:" + mqProducerGroup);
    logger.info("namesrvAddr:" + namesrvAddr);
    if (PRODUCERS.containsKey(mqProducerGroup)) {
      defaultMQProducer = PRODUCERS.get(mqProducerGroup);
      logger.info(mqProducerGroup + " 的Producer已经被创建，直接返回，建议检查配置是否重复");
      return;
    }
    // 初始化
    defaultMQProducer = new DefaultMQProducer(mqProducerGroup);
    if (PRODUCERS.putIfAbsent(mqProducerGroup, defaultMQProducer) != null) {
      defaultMQProducer = PRODUCERS.get(mqProducerGroup);
      logger.info(mqProducerGroup + " 的Producer已经被创建，直接返回，建议检查配置是否重复");
      return;
    }
    defaultMQProducer.setUnitName(rocketMqName + "_" + mqProducerGroup);
    defaultMQProducer.setNamesrvAddr(namesrvAddr);
    defaultMQProducer.setVipChannelEnabled(false);
    defaultMQProducer.start();
    Thread.sleep(500);
    isStarted = true;
    synchronized (this) {
      this.notifyAll();
    }
//        KangarooDiscovery.start(zookeeperAddress);
    logger.info("DefaultMQProducer start success!");

  }


  /**
   * 发送消息操作，在producer init完成之后调用 step： 1，通过RocketMq发送消息 2，如果失败，把消息发到消息系统，由消息系统进行重发
   */
  public DubboResult sendMessage(final RocketMessageDto msgDto) {
    DubboResult result = validateProducerClientStatus();
    if (!result.isSuccess()) {
      return result;
    }
    if (msgDto == null) {
      logger.error("msgDto不能为null。");
      return DubboResultBuilder.error(DubboResultCodeEnums.PARAM_ERROR.value(), "msgDto不能为null。");
    }
    saveTopic(msgDto.getTopic());
    StopWatch stopWatch = new StopWatch();
    try {
      logger.info("开始发送消息，消息为：{}, 业务key为:{} topic为:{}, tags为:{}",
          JSONObject.toJSONString(msgDto.getData()), msgDto.getKey(), msgDto.getTopic(),
          msgDto.getTags());
      SendResult sendResult = defaultMQProducer.send(msgDto.getMessage());
      if (sendResult != null && sendResult.getSendStatus() != SendStatus.SEND_OK) {
        logger.error("发送消息状态有错:" + JSONObject.toJSONString(sendResult));
        return DubboResultBuilder.success(sendResult.getSendStatus());
      }
      return DubboResultBuilder.success(null);
    } catch (Exception e) {
      //发送消息失败，消息发送至消息系统记录失败消息，由消息系统重发
      logger.error("通过Rocket发送消息失败,将消息保存至消息中心。", e);
      return saveToMessageCenter(msgDto);
    } finally {
      logger.info("rocketmq mq 发送消息结束， 时间为：{}ms", stopWatch.elapsedTime());
    }
  }

  private DubboResult saveToMessageCenter(RocketMessageDto msgDto) {
    MessageDto messageDto = new MessageDto();
    String requestId = RequestIdUtils.generateNextRequestId();
    messageDto.setReceiverParam(JsonSerializeUtil.serialize(msgDto.getData()));
    messageDto.setReceiverAddress(rocketMqName);
    messageDto.setSourcePlatform(PlatformConfig.getPlatform());
    messageDto.setDeliverType(DeliverTypeConstants.ROCKET_MQ);
    messageDto.setKeyword(msgDto.getKey());
    messageDto.setTopic(msgDto.getTopic());
    messageDto.setTags(msgDto.getTags());
    messageDto.setRequestId(requestId);
    messageDto.setClientIp(localIp);
    return producerService.sendMessage(messageDto);
  }

  private void saveTopic(String topic) {
    //如果发送的topic第一次发送则发送给消息系统
    if (!TOPICS.contains(topic) && TOPICS.add(topic)) {
      new Thread(new Runnable() {
        @Override
        public void run() {
          RocketProducerTopicDto topicDto = new RocketProducerTopicDto();
          topicDto.setProducerGroup(mqProducerGroup);
          topicDto.setRocketMqName(rocketMqName);
          topicDto.setTopic(topic);
          producerService.saveTopic(topicDto);
        }
      }).start();
    }
  }

  private DubboResult validateProducerClientStatus() {
    //如果发送消息时，producerClient还未启动
    if (!isStarted) {
      try {
        synchronized (this) {
          int count = 20;
          while (!isStarted && count > 0) {
            count--;
            if (startError) {
              return DubboResultBuilder
                  .error(DubboResultCodeEnums.INTERNAL_ERROR.value(), "rocketMqProducer启动失败，请检查日志");
            }
            this.wait(2000);
          }
        }
      } catch (InterruptedException e) {
        logger.error("发送消息等待初始化错误", e);
        throw new RuntimeException(e);
      }
    }
    if (startError) {
      logger.error("rocketMqProducer启动失败，请检查启动日志是否有异常。");
      return DubboResultBuilder.error(DubboResultCodeEnums.INTERNAL_ERROR.value(), "rocketMqProducer启动失败，请检查日志");
    }
    return DubboResultBuilder.success();
  }

  /**
   * Spring bean destroy-method 生产者释放，在消息发送完成之后操作
   */
  public void destroy() {
    try {
      if (defaultMQProducer != null) {
        defaultMQProducer.shutdown();
      }
    } catch (Throwable t) {
      logger.error("RocketMq 关闭异常", t);
    }

  }

  public void setMqProducerGroup(String mqProducerGroup) {
    this.mqProducerGroup = mqProducerGroup;
  }

  public void setRocketMqName(String rocketMqName) {
    this.rocketMqName = rocketMqName;
  }

  public void setProducerService(ProducerService producerService) {
    this.producerService = producerService;
  }

  public void setRocketMqAddrService(RocketMqAddrService rocketMqAddrService) {
    this.rocketMqAddrService = rocketMqAddrService;
  }

  @Override
  public void onApplicationEvent(ContextRefreshedEvent contextRefreshedEvent) {
    Timer timer = new Timer();
    timer.schedule(new TimerTask() {
      @Override
      public void run() {
        try {
          init();
        } catch (Exception e) {
          logger.error("defaultMqProducer启动失败", e);
          startError = true;
          //每隔3秒重新启动producerClient
          for (int i = 0; i < 5 && !isStarted; i++) {
            reInit(i);

          }
          if (startError) {
            logger.error("5次重新启动producerClient:" + mqProducerGroup + "仍然失败");
          }
        }
      }

      private void reInit(int i) {
        try {
          Thread.sleep(3000);
          init();
          startError = false;
          logger.info("重新启动producerClient" + mqProducerGroup + ", 重新启动成功");
        } catch (Exception e1) {
          logger.error(mqProducerGroup + "第" + (i + 1) + "次重新启动失败", e1);
        }
      }
    }, 1000 * 2);

  }
}
