/*
 * Decompiled with CFR 0.152.
 */
package com.google.datastore.v1.client;

import com.google.api.client.http.GenericUrl;
import com.google.api.client.http.HttpRequestFactory;
import com.google.api.client.http.HttpResponse;
import com.google.api.client.util.Preconditions;
import com.google.datastore.v1.client.Datastore;
import com.google.datastore.v1.client.DatastoreEmulatorException;
import com.google.datastore.v1.client.DatastoreEmulatorOptions;
import com.google.datastore.v1.client.RemoteRpc;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.file.Files;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

public class DatastoreEmulator
extends Datastore {
    private static final int STARTUP_TIMEOUT_SECS = 30;
    private final String host;
    private final DatastoreEmulatorOptions options;
    private volatile State state = State.NEW;
    private File projectDirectory;

    DatastoreEmulator(RemoteRpc rpc, String localHost, DatastoreEmulatorOptions options) {
        super(rpc);
        this.host = "http://" + localHost;
        this.options = options;
    }

    public void clear() throws DatastoreEmulatorException {
        HttpRequestFactory client = this.remoteRpc.getHttpRequestFactory();
        try {
            GenericUrl url = new GenericUrl(this.host + "/reset");
            HttpResponse httpResponse = client.buildPostRequest(url, null).execute();
            if (!httpResponse.isSuccessStatusCode()) {
                throw new DatastoreEmulatorException("POST request to /reset returned HTTP status " + httpResponse.getStatusCode());
            }
        }
        catch (IOException e) {
            throw new DatastoreEmulatorException("Exception trying to clear the emulator", e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized void start(String emulatorDir, String projectId, String ... commandLineOptions) throws DatastoreEmulatorException {
        Preconditions.checkNotNull((Object)emulatorDir, (Object)"emulatorDir cannot be null");
        Preconditions.checkNotNull((Object)projectId, (Object)"projectId cannot be null");
        Preconditions.checkState((this.state == State.NEW ? 1 : 0) != 0, (Object)"Cannot call start() more than once.");
        try {
            this.startEmulatorInternal(emulatorDir, projectId, commandLineOptions);
            this.state = State.STARTED;
        }
        finally {
            if (this.state != State.STARTED) {
                this.state = State.STOPPED;
            }
        }
    }

    void startEmulatorInternal(String emulatorDir, String projectId, String ... commandLineOptions) throws DatastoreEmulatorException {
        Process emulatorStartProcess;
        this.projectDirectory = this.createProjectDirectory(emulatorDir, projectId);
        ArrayList<String> cmd = new ArrayList<String>(Arrays.asList("./cloud_datastore_emulator", "start", "--testing"));
        Collections.addAll(cmd, commandLineOptions);
        cmd.add(this.projectDirectory.getPath());
        try {
            emulatorStartProcess = this.newEmulatorProcess(emulatorDir, cmd).start();
        }
        catch (IOException e) {
            throw new DatastoreEmulatorException("Could not start emulator", e);
        }
        Runtime.getRuntime().addShutdownHook(new Thread(){

            @Override
            public void run() {
                emulatorStartProcess.destroy();
            }
        });
        StartupMonitor monitor = new StartupMonitor(emulatorStartProcess.getInputStream());
        try {
            monitor.start();
            if (!monitor.startupCompleteLatch.await(30L, TimeUnit.SECONDS)) {
                throw new DatastoreEmulatorException("Emulator did not start within 30 seconds");
            }
            if (!monitor.success) {
                throw new DatastoreEmulatorException("Emulator did not start normally");
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new DatastoreEmulatorException("Received an interrupt", e);
        }
    }

    private File createProjectDirectory(String emulatorDir, String projectId) throws DatastoreEmulatorException {
        File projectDirectory;
        try {
            projectDirectory = Files.createTempDirectory("datastore-emulator", new FileAttribute[0]).toFile();
        }
        catch (IOException e) {
            throw new DatastoreEmulatorException("Could not create temporary project directory", e);
        }
        List<String> cmd = Arrays.asList("./cloud_datastore_emulator", "create", "--project_id=" + projectId, projectDirectory.getPath());
        try {
            int retCode = this.newEmulatorProcess(emulatorDir, cmd).start().waitFor();
            if (retCode != 0) {
                throw new DatastoreEmulatorException(String.format("Could not create project (retcode=%d)", retCode));
            }
        }
        catch (IOException e) {
            throw new DatastoreEmulatorException("Could not create project", e);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new DatastoreEmulatorException("Received an interrupt", e);
        }
        return projectDirectory;
    }

    private ProcessBuilder newEmulatorProcess(String emulatorDir, List<String> cmd) {
        ProcessBuilder builder = new ProcessBuilder(cmd);
        builder.directory(new File(emulatorDir));
        builder.redirectErrorStream(true);
        builder.environment().putAll(this.options.getEnvVars());
        return builder;
    }

    public synchronized void stop() throws DatastoreEmulatorException {
        this.stopEmulatorInternal();
        if (this.state != State.STOPPED) {
            this.state = State.STOPPED;
            if (this.projectDirectory != null) {
                try {
                    Process process = new ProcessBuilder("rm", "-r", this.projectDirectory.getAbsolutePath()).start();
                    if (process.waitFor() != 0) {
                        throw new IOException("Temporary project directory deletion exited with " + process.exitValue());
                    }
                }
                catch (IOException e) {
                    throw new IllegalStateException("Could not delete temporary project directory", e);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw new IllegalStateException("Could not delete temporary project directory", e);
                }
            }
        }
    }

    public synchronized File getProjectDirectory() {
        Preconditions.checkState((this.state == State.STARTED ? 1 : 0) != 0);
        return this.projectDirectory;
    }

    void stopEmulatorInternal() throws DatastoreEmulatorException {
        HttpRequestFactory client = this.remoteRpc.getHttpRequestFactory();
        GenericUrl url = new GenericUrl(this.host + "/shutdown");
        try {
            HttpResponse httpResponse = client.buildPostRequest(url, null).execute();
            if (!httpResponse.isSuccessStatusCode()) {
                throw new DatastoreEmulatorException("POST request to /shutdown returned HTTP status " + httpResponse.getStatusCode());
            }
        }
        catch (IOException e) {
            throw new DatastoreEmulatorException("Exception trying to stop the emulator", e);
        }
    }

    static class StartupMonitor
    extends Thread {
        private final InputStream inputStream;
        private volatile boolean success = false;
        private final CountDownLatch startupCompleteLatch = new CountDownLatch(1);

        StartupMonitor(InputStream inputStream) {
            this.inputStream = inputStream;
            this.setDaemon(true);
        }

        @Override
        public void run() {
            try {
                String line;
                BufferedReader br = new BufferedReader(new InputStreamReader(this.inputStream));
                while ((line = br.readLine()) != null) {
                    System.out.println(line);
                    if (this.success || !line.contains("Dev App Server is now running")) continue;
                    this.success = true;
                    this.startupCompleteLatch.countDown();
                }
            }
            catch (IOException ioe) {
                if (!this.success) {
                    System.err.println("Received an IOException before emulator startup completed. Emulator is in an unknown state.");
                } else {
                    System.err.println("Received an exception handling output from the emulator. Logging will stop but the emulator is probably ok.");
                }
                ioe.printStackTrace();
            }
            finally {
                if (!this.success) {
                    this.startupCompleteLatch.countDown();
                }
            }
        }
    }

    static enum State {
        NEW,
        STARTED,
        STOPPED;

    }
}

